/* linux/drivers/char/mity_gpio.c 

   Mity Mite GPIO device driver
   Copyright (c) 2003 Renzo Davoli <renzo@cs.unibo.it> 

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.


   minor devices:
   mknod /dev/mitygpio07 c xxx 0 	read or write GPIO-7..0 as a whole
   mknod /dev/mitygpio8F c xxx 1 	read or write GPIO-15..8 as a whole
   mknod /dev/mitygpiomask07 c xxx 2	write GPIO-7..0 mask: FF all outputs
   							00 all inputs
   mknod /dev/mitygpiomask8F c xxx 3	write GPIO-15..8 mask: FF all outputs
   							00 all inputs
   mknod /dev/mitygpio0 c xxx 16	rw GPIO-0 (write lsb, reads '0' or '1')
   mknod /dev/mitygpio1 c xxx 17	rw GPIO-1
   mknod /dev/mitygpio2 c xxx 18	....
   mknod /dev/mitygpio3 c xxx 19
   mknod /dev/mitygpio4 c xxx 20
   mknod /dev/mitygpio5 c xxx 21
   mknod /dev/mitygpio6 c xxx 22
   mknod /dev/mitygpio7 c xxx 23
   mknod /dev/mitygpio8 c xxx 24
   mknod /dev/mitygpio9 c xxx 25
   mknod /dev/mitygpioA c xxx 26
   mknod /dev/mitygpioB c xxx 27
   mknod /dev/mitygpioC c xxx 28
   mknod /dev/mitygpioD c xxx 29
   mknod /dev/mitygpioE c xxx 30
   mknod /dev/mitygpioF c xxx 31
   mknod /dev/mitygpiomask0 c xxx 32	write GPIO-0 mask ('0'=r,'1'=w)
   mknod /dev/mitygpiomask1 c xxx 33
   mknod /dev/mitygpiomask2 c xxx 34
   mknod /dev/mitygpiomask3 c xxx 35
   mknod /dev/mitygpiomask4 c xxx 36
   mknod /dev/mitygpiomask5 c xxx 37
   mknod /dev/mitygpiomask6 c xxx 38
   mknod /dev/mitygpiomask7 c xxx 39
   mknod /dev/mitygpiomask8 c xxx 40
   mknod /dev/mitygpiomask9 c xxx 41
   mknod /dev/mitygpiomaskA c xxx 42
   mknod /dev/mitygpiomaskB c xxx 43
   mknod /dev/mitygpiomaskC c xxx 44
   mknod /dev/mitygpiomaskD c xxx 45
   mknod /dev/mitygpiomaskE c xxx 46
   mknod /dev/mitygpiomaskF c xxx 47

*/

#include <linux/config.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <asm/io.h>

// serve? #include <linux/mity_gpio.h>

#define MITY_GPIO_READ_A 0x46
#define MITY_GPIO_WRITE_A 0x47
#define MITY_GPIO_READ_B 0x4C
#define MITY_GPIO_WRITE_B 0x4D
#define MITY_GPIO_MASK_A 0x4E
#define MITY_GPIO_MASK_B 0x4F
#define MITY_GPIO_COMMAND 0x22
#define MITY_GPIO_DATA 0x23
#define MITY_REG_SET 0x13
#define MITY_REG_SET_OPEN 0xc5
#define MITY_REG_SET_CLOSE 0x00

#define NAME "mity_gpio"

MODULE_AUTHOR("Renzo Davoli <renzo@cs.unibo.it>");
MODULE_DESCRIPTION("Mity-Mite GPIO access v.1.0");
MODULE_LICENSE("GPL");

static int major = 0;		/* default to dynamic major */
MODULE_PARM(major, "i");
MODULE_PARM_DESC(major, "Major device number");

static char mity_data_A, mity_data_B, mity_mask_A, mity_mask_B;

static ssize_t mity_gpio_write(struct file *file, const char *data, 
				 size_t len, loff_t *ppos)
{
	unsigned m = minor(file->f_dentry->d_inode->i_rdev);
	char i;

	if (ppos != &file->f_pos)
		return -ESPIPE;

	char c;
	if (get_user(c, data+len-1))
		return -EFAULT;
	
	// spinlock? mity is not SMP!

	switch(m) {
		case 0:
			outb(MITY_GPIO_WRITE_A, MITY_GPIO_COMMAND);
			outb(c, MITY_GPIO_DATA);
			mity_data_A=c;
			break;
		case 1:
			outb(MITY_GPIO_WRITE_B, MITY_GPIO_COMMAND);
			outb(c, MITY_GPIO_DATA);
			mity_data_B=c;
			break;
		case 2:
			outb(MITY_GPIO_MASK_A, MITY_GPIO_COMMAND);
			outb(c, MITY_GPIO_DATA);
			mity_mask_A=c;
			break;
		case 3:
			outb(MITY_GPIO_MASK_B, MITY_GPIO_COMMAND);
			outb(c, MITY_GPIO_DATA);
			mity_mask_B=c;
			break;
		case 16:
		case 17:
		case 18:
		case 19:
		case 20:
		case 21:
		case 22:
		case 23:
			i=m-16;
			if(c%2)
				mity_data_A |= 1<<i;
			else
				mity_data_A &= ~(1<<i);
			outb(MITY_GPIO_WRITE_A, MITY_GPIO_COMMAND);
			outb(mity_data_A, MITY_GPIO_DATA);
			break;
		case 24:
		case 25:
		case 26:
		case 27:
		case 28:
		case 29:
		case 30:
		case 31:
			i=m-24;
			if(c%2)
				mity_data_B |= 1<<i;
			else
				mity_data_B &= ~(1<<i);
			outb(MITY_GPIO_WRITE_B, MITY_GPIO_COMMAND);
			outb(mity_data_B, MITY_GPIO_DATA);
			break;
		case 32:
		case 33:
		case 34:
		case 35:
		case 36:
		case 37:
		case 38:
		case 39:
			i=m-32;
			if(c%2)
				mity_mask_A |= 1<<i;
			else
				mity_mask_A &= !(1<<i);
			outb(MITY_GPIO_MASK_A, MITY_GPIO_COMMAND);
			outb(mity_mask_A, MITY_GPIO_DATA);
			break;
		case 40:
		case 41:
		case 42:
		case 43:
		case 44:
		case 45:
		case 46:
		case 47:
			i=m-40;
			if(c%2)
				mity_mask_B |= 1<<i;
			else
				mity_mask_B &= !(1<<i);
			outb(MITY_GPIO_MASK_B, MITY_GPIO_COMMAND);
			outb(mity_mask_B, MITY_GPIO_DATA);
			break;
	}	
	return len;
}

static ssize_t mity_gpio_read(struct file *file, char *buf,
				size_t len, loff_t *ppos)
{
	unsigned m = minor(file->f_dentry->d_inode->i_rdev);
	int value=0;
	char i;

	if (ppos != &file->f_pos)
		return -ESPIPE;

	switch(m) {
		case 0:
			outb(MITY_GPIO_READ_A, MITY_GPIO_COMMAND);
			value=inb(MITY_GPIO_DATA);
			break;
		case 1:
			outb(MITY_GPIO_READ_B, MITY_GPIO_COMMAND);
			value=inb(MITY_GPIO_DATA);
			break;
		case 16:
		case 17:
		case 18:
		case 19:
		case 20:
		case 21:
		case 22:
		case 23:
			i=m-16;
			outb(MITY_GPIO_READ_A, MITY_GPIO_COMMAND);
			value=(inb(MITY_GPIO_DATA) & 1<<i)?'1':'0';
			break;
		case 24:
		case 25:
		case 26:
		case 27:
		case 28:
		case 29:
		case 30:
		case 31:
			i=m-24;
			outb(MITY_GPIO_READ_B, MITY_GPIO_COMMAND);
			value=(inb(MITY_GPIO_DATA) & 1<<i)?'1':'0';
			break;
	}
	if (put_user(value, buf))
		return -EFAULT;
	
	return 1;
}

#define MITY_REG_SET 0x13
#define MITY_REG_SET_OPEN 0xc5
#define MITY_REG_SET_CLOSE 0xc5
static int mity_gpio_open(struct inode *inode, struct file *file)
{
	unsigned m = minor(inode->i_rdev);
	if (m > 3 && m < 16 || m > 47)
		return -EINVAL;

	return 0;
}

static int mity_gpio_release(struct inode *inode, struct file *file)
{
	return 0;
}


static struct file_operations mity_gpio_fops = {
	.owner   = THIS_MODULE,
	.write   = mity_gpio_write,
	.read    = mity_gpio_read,
	.open    = mity_gpio_open,
	.release = mity_gpio_release,
};

static int __init mity_gpio_init(void)
{
	int r;

	printk(KERN_DEBUG NAME ": Mity-Mite GPIO driver v.1.0 (C) 2003 R.Davoli\n");

	r = register_chrdev(major, NAME, &mity_gpio_fops);
	if (r < 0) {
		printk(KERN_ERR NAME ": unable to register character device\n");
		return r;
	}
	if (!major) {
		major = r;
		printk(KERN_DEBUG NAME ": got dynamic major %d\n", major);
		outb(MITY_REG_SET, MITY_GPIO_COMMAND);
		outb(MITY_REG_SET_OPEN, MITY_GPIO_DATA);
		mity_mask_A=mity_mask_B=0;
		outb(MITY_GPIO_MASK_A, MITY_GPIO_COMMAND);
		outb(mity_mask_A, MITY_GPIO_DATA);
		outb(MITY_GPIO_MASK_B, MITY_GPIO_COMMAND);
		outb(mity_mask_B, MITY_GPIO_DATA);
	}

	return 0;
}

static void __exit mity_gpio_cleanup(void)
{
	outb(MITY_REG_SET, MITY_GPIO_COMMAND);
	outb(MITY_REG_SET_CLOSE, MITY_GPIO_DATA);
	unregister_chrdev(major, NAME);
}

module_init(mity_gpio_init);
module_exit(mity_gpio_cleanup);

/*
    Local variables:
        compile-command: "make -k -C ../.. SUBDIRS=drivers/char modules"
        c-basic-offset: 8
    End:
*/
